#ifndef XFLTK__WIDGETCREATOR_H_
#define XFLTK__WIDGETCREATOR_H_

#include <map>
#include <fltk/widget.h>

#include "utils.h"

#include "GUI.h"

namespace xfltk {

/**
 * Construct widget and set it attributes
 */
class WidgetCreator {
	
	GUI& _gui;
	
	public:
	
	WidgetCreator(GUI& gui): _gui(gui) {};
	
	GUI& gui() { return _gui; }
	
	/**
	 * This method create and initialize widget with given name and atributes.
	 * It is invoked before any children widgets are create.
	 * @param name
	 * @param attr widget atributes (null-teriminated array)
	 * 		(attr[i], attr[i+1]) = (attribute name, attribute value) when i = 0, 2, 4, ...
	 * @return pointer to created widget or <code>0</code> (when there is no widget creator for given name)
	 */
	virtual fltk::Widget* create_begin(const char** attr) = 0;
	
	//virtual void before_child_create();
	
	//virtual void after_child_create();
	
	/**
	 * End constructing widget with given name.
	 * @param name
	 */
	virtual void create_end() = 0;
	
	virtual ~WidgetCreator() {};
	
};

/**
 * Construct widget creator witch can create widget and set its attributes.
 * It includes static map (register): WidgetFactory name string -> WidgetFactory objects 
 */
class WidgetFactory {
	
	typedef std::map<const char*, WidgetFactory*, utils::ltstr> widget_factories_map;
	
	/**
	 * Widget WidgetFactory name string -> WidgetFactory objects map
	 */
	static widget_factories_map widget_factories;
	
	protected:
	
	/**
	 * Creates WidgetCreators and registers it under <code>name</code>.
	 * @param name widget name
	 */
	WidgetFactory(const char* name);
	
	public:
	
	/**
	 * Find creator with given <code>name</code> in register.
	 * @param name
	 * @return pointer to finded widget or <code>0</code> (when there is no widget creator for given name)
	 */
	static WidgetFactory* find(const char* name);
	
	/**
	 * @return widget creator witch can create widget with given name
	 */
	static WidgetCreator* getCreator(const char* name, GUI& gui);
	
	/**
	 * This method should create and initialize widget.
	 * It is invoked before any children widgets are create.
	 * @param attr widget atributes (null-teriminated array)
	 * 		(attr[i], attr[i+1]) = (attribute name, attribute value) when i = 0, 2, 4, ...
	 * @return pointer to created widget or <code>0</code> (when there is no widget creator for given name)
	 */
	virtual WidgetCreator* getCreator(GUI& gui) = 0;
	
	/**
	 * Do nothing.
	 */
	virtual ~WidgetFactory() {};
	
};

/**
 * Set specific property to given <code>value</code>
 */
typedef void property_setter(fltk::Widget* obj, const char* value, WidgetCreator& creator);

/**
 * Factory with property_setter map for specyfic widget type
 */
class HierarhyWidgetFactory: public WidgetFactory {

protected:

	typedef std::map<const char*, property_setter*, utils::ltstr> widget_property_setters_map;

private:

	/**
	 * Parent or <code>0</code> if there is root
	 */
	HierarhyWidgetFactory* parent;

	/**
	 * Property name string -> property setter function map
	 */
	widget_property_setters_map widget_property_setters;

protected:

	/**
	 * Constructor.
	 * @param name
	 * @param parent
	 */
	HierarhyWidgetFactory(const char* name, HierarhyWidgetFactory* parent);
	
	/**
	 * @return property setter map
	 */
	widget_property_setters_map& getWidgetPropertySetters();
	
public:
			
	/*
	 * Fill property setter map.
	 */
	//virtual void fill_property_setters() { };
	
	void setProperty(fltk::Widget* obj, const char* name, const char* value, WidgetCreator& creator);
	
	void setProperties(fltk::Widget* obj, const char** attr, WidgetCreator& creator);
	
	void addPropertySetter(const char* name, property_setter setter);
	
};

/**
 * 
 */
class HierarhyWidgetCreatorBase: public WidgetCreator {
	
	protected:
	
	HierarhyWidgetFactory& factory;
	
	public:
	
	virtual fltk::Widget* new_widget() = 0;
	
	HierarhyWidgetCreatorBase(GUI& gui, HierarhyWidgetFactory& factory): WidgetCreator(gui), factory(factory) {}
	
	virtual fltk::Widget* create_begin(const char** attr) {
		fltk::Widget* result = new_widget();
		factory.setProperties(result, attr, *this);
		return result;
	}
	
	virtual void create_end() {};	
	
};

template <typename fltkWidgetType>
struct HierarhyWidgetCreator0: public HierarhyWidgetCreatorBase {
	
	HierarhyWidgetCreator0<fltkWidgetType>(GUI& gui, HierarhyWidgetFactory& factory): HierarhyWidgetCreatorBase(gui, factory) {}
	
	virtual fltk::Widget* new_widget() {
		return new fltkWidgetType();
	}
	
};

template <typename fltkWidgetType>
struct HierarhyWidgetCreator: public HierarhyWidgetCreatorBase {
	
	HierarhyWidgetCreator<fltkWidgetType>(GUI& gui, HierarhyWidgetFactory& factory): HierarhyWidgetCreatorBase(gui, factory) {}
	
	virtual fltk::Widget* new_widget() {
		return new fltkWidgetType(0, 0, 0, 0);
	}
	
};

} //namespace xfltk

#endif /*XFLTK__WIDGETCREATOR_H_*/
