#include "GUI.h"

#include "WidgetCreator.h"

#include "utils.h"

#include <fstream>
#include <stack>
#include <sstream>

namespace xfltk {

struct ParserInfo {
	
	GUI& gui;
	
	XML_Parser& parser;
	
	std::stack<WidgetCreator*> creators_stack;
	
	/**
	 * Ilość przeanalizowanych błędów
	 */
	unsigned errors_len;
	
	unsigned skip_tag;
	
	ParserInfo(GUI& gui, XML_Parser& parser): gui(gui), parser(parser), errors_len(0u), skip_tag(0u) {};
	
	/**
	 * @return czy przetwarzać taga?
	 */
	bool tag_begin() {
		if (skip_tag == 0u) return true;
		++skip_tag;
		return false;
	};
	
	/**
	 * @return czy przetwarzać taga?
	 */
	bool tag_end() {
		if (skip_tag != 0u) {
			--skip_tag;
			return false;
		}
		return true;
	};
	
	/**
	 * Pomiń tag-a wwraz z dziećmi
	 */
	void tag_skip() {
		++skip_tag;
	};
	
};

void XMLCALL
startElement(void *userData, const char *name, const char **atts) {
	ParserInfo* p = (ParserInfo*)userData;
	if (!p->tag_begin()) return;
	WidgetCreator* wc = WidgetFactory::getCreator(name, p->gui);
	if (!wc) {	//error creating tag
		p->gui.errors.append(new InvalidTagError(name, XML_GetCurrentLineNumber(p->parser), Error::skip_tag));
		p->tag_skip();
		return;
	}
	const char* id = utils::find_pair(atts, "id");
	p->creators_stack.push(wc);
	fltk::Widget* w = wc->create_begin(atts);
	if (!p->gui.root())
		p->gui._root = w;
	if (id)
		p->gui.setWidget(utils::strcopy(id), w);
}

static void XMLCALL
endElement(void *userData, const char *name) {
	ParserInfo* p = (ParserInfo*)userData;
	if (!p->tag_end()) return;
	p->creators_stack.top()->create_end();
	delete p->creators_stack.top();
	p->creators_stack.pop();
}

GUI::GUI(std::istream& in_stream): _root(0) {
	in_stream >> *this;
}

GUI::GUI(const char* file_name): _root(0) {
	readFromFile(file_name);
}

void GUI::clear() {
	for (widget_map_type::iterator i = widget_map.begin(); i != widget_map.end(); ++i)
		delete[] i->first;
	widget_map.clear();
	delete _root;
	_root = 0;
}

GUI::~GUI() {
	clear();
}

fltk::Widget* GUI::operator[](const char* name) {
	widget_map_type::iterator res = widget_map.find(name);
	return (res == widget_map.end()) ? 0 : res->second;
}

std::istream& operator>>(std::istream& in_stream, GUI& g) {
	g.clear();
	g.errors.clear();
//	std::stack<WidgetCreator*> creators_stack;
//	std::pair<GUI*, std::stack<WidgetCreator*>*> p(&g, &creators_stack);
	XML_Parser parser = XML_ParserCreate(NULL);
	ParserInfo parser_info(g, parser);

	void *buff = XML_GetBuffer(parser, BUFSIZ);
	if (buff == NULL) {
    	/* TODO handle error */
  	}	
	int done;
	XML_SetUserData(parser, &parser_info);
	XML_SetElementHandler(parser, startElement, endElement);
	
	do {
    	size_t len = in_stream.read((char*)buff, BUFSIZ).gcount();
    	
    	done = len < sizeof(buff);
    	if (XML_ParseBuffer(parser, len, done) == XML_STATUS_ERROR) {
    		g.errors.append(
    		   new GeneralStaticError( XML_ErrorString(XML_GetErrorCode(parser)),
      								   XML_GetCurrentLineNumber(parser), Error::skip_all)
      		);
      		return in_stream;
		}
	} while (!done);
	
	XML_ParserFree(parser);
	return in_stream;
}

void GUI::readFromFile(const char* file_name) {
	std::ifstream file(file_name);
	/*if (!file.is_open()) {
		errors.clear();
		errors.append();
	}*/
	file >> *this;
}

} //namespace xfltk
