#ifndef XFLTK__ERROR_H_
#define XFLTK__ERROR_H_

#include <exception>
#include <ostream>
#include <vector>

#include "utils.h"

namespace xfltk {

class Error: public std::exception {
	
	public:
	
		enum ParserAction {
			ignor = 0, skip_arg = 1, skip_tag = 1 + 2, skip_all = 1 + 2 + 4
		};
		
		explicit Error(int line_nr, ParserAction parser_action) throw() : _line_nr(line_nr), _parser_action(parser_action) {}
		
		virtual ParserAction parser_action() const { return _parser_action; };
		
		bool parser_skip_arg() { return parser_action() & skip_arg; }
		
		bool parser_skip_tag() { return parser_action() & skip_tag; }
		
		bool parser_skip_all() { return parser_action() & skip_all; }
		
		int line_nr() const { return _line_nr; }
		
		friend std::ostream& operator<<(std::ostream& out_stream, const Error& error);
		
	protected:
	
		int _line_nr;
		
		ParserAction _parser_action;
};

class GeneralStaticError: public Error {
	
	protected:
	
	const char* _static_text;
	
	public:
	
	GeneralStaticError(const char* static_text, int line_nr, ParserAction parser_action): Error(line_nr, parser_action), _static_text(static_text) {}
	
	const char* what() const throw() { return _static_text; }
	
};

class GeneralDynamicError: public Error {
	
	protected:
	
	const char* _dynamic_text;
	
	public:
	
	GeneralDynamicError(const char* dynamic_text, int line_nr, ParserAction parser_action);
	
	GeneralDynamicError(const GeneralDynamicError& to_copy);
	
	virtual ~GeneralDynamicError() throw() { delete[] _dynamic_text; }
	
	const char* what() const throw() { return _dynamic_text; }
	
};

class InvalidTagError: public GeneralDynamicError {
	
	public:
	
	static const char* make_text(const char* tag_name);
	
	InvalidTagError(const char* tag_name, int line_nr, ParserAction parser_action);
	
};

class Errors {
	
	typedef std::vector<Error*> error_vec;
	
	std::vector<Error*> errors;
	
	public:
	
	typedef error_vec::size_type size_type;
	
	~Errors() { clear(); };
	
	size_type size() const;
	
	bool empty() const;
	
	const Error& operator[](size_type n) const;
		
	void append(Error* error);
	
	void clear();
	
	friend std::ostream& operator<<(std::ostream& out_stream, const Errors& errors);
	
};

};

#endif /*XFLTK__ERROR_H_*/
